<?php
namespace App\Livewire;

use Livewire\Component;
use App\Models\Tarea;
use App\Models\TareaEstado;

class ListaTareas extends Component
{
    public $titulo, $descripcion, $tarea_estado_id, $fecha_entrega;
    public $modo_edicion = false;
    public $tarea_id_editar;
    public $tareas = [];

    protected $listeners = ['confirmarFinalizar' => 'finalizar'];

    public function mount()
    {
        $this->tarea_estado_id = 1; // ID del estado "Por realizar"
        $this->refrescarTareas();
    }


    public function guardar()
    {
        $this->validate([
            'titulo' => 'required|string|max:255',
            'descripcion' => 'nullable|string',
            'tarea_estado_id' => 'required|exists:tarea_estados,id',
            'fecha_entrega' => 'nullable|date',
        ]);

        if ($this->modo_edicion && $this->tarea_id_editar) {
            $tarea = Tarea::find($this->tarea_id_editar);
            $tarea->update([
                'titulo' => $this->titulo,
                'descripcion' => $this->descripcion,
                'tarea_estado_id' => $this->tarea_estado_id,
                'fecha_entrega' => $this->fecha_entrega,
            ]);
        } else {
            Tarea::create([
                'titulo' => $this->titulo,
                'descripcion' => $this->descripcion,
                'tarea_estado_id' => $this->tarea_estado_id,
                'fecha_entrega' => $this->fecha_entrega,
            ]);
        }

        $this->cancelarEdicion();
        $this->refrescarTareas();
    }

    public function editar($id)
    {
        $tarea = Tarea::findOrFail($id);
        $this->titulo = $tarea->titulo;
        $this->descripcion = $tarea->descripcion;
        $this->tarea_estado_id = $tarea->tarea_estado_id;
        $this->fecha_entrega = $tarea->fecha_entrega;
        $this->tarea_id_editar = $tarea->id;
        $this->modo_edicion = true;
    }

    public function cancelarEdicion()
    {
        $this->reset(['titulo', 'descripcion', 'tarea_estado_id', 'fecha_entrega', 'modo_edicion', 'tarea_id_editar']);
    }

    public function refrescarTareas()
    {
        $this->tareas = Tarea::with('estado')->latest()->get();
    }

    public function render()
    {
        return view('livewire.lista-tareas', [
            'estados' => TareaEstado::all(),
        ]);
    }

    public function finalizar($id)
    {
        logger('Evento FINALIZAR recibido con ID: ' . $id);

        $tarea = Tarea::findOrFail($id);
        $finalizado = TareaEstado::where('estado', 'Finalizada')->first();

        if ($finalizado) {
            $tarea->tarea_estado_id = $finalizado->id;
            $tarea->save();
            $this->refrescarTareas();
            logger('Tarea finalizada correctamente');
        } else {
            logger('⚠️ No se encontró estado "Finalizada"');
        }
    }
}
